<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}
/**
* @package PaytalkKopokopoWoocommerce
*/
/*
Plugin Name: KopoKopo Lipa Na Mpesa
Plugin URI: https://wordpress.org/plugins/paytalk-kopokopo-gateway/
Description: KopoKopo Lipa Na Mpesa is a WooCommerce extension plugin that allows website owners to receive payment via Mpesa Paybill/Till Number. It uses KopoKopo APIs (K2-Connect) to process payments. The plugin has been developed by <a href='https://paytalk.co.ke' target='_blank'>paytalk.co.ke.</a>
Version: 3.1.1
Author: Paytalk.co.ke
Author URI: https://paytalk.co.ke
License: GNU General Public License v3.0
License URI: http://www.gnu.org/licenses/gpl-3.0.html
Text Domain: woo-paytalk-kopokopo-gateway
WC requires at least: 4.0.0
WC tested up to: 9.8
*/

if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) || class_exists( 'WooCommerce' ) ) {

    // HPOS compatibility declaration
    add_action( 'before_woocommerce_init', function() {
        if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
                'custom_order_tables',
                __FILE__,
                true
            );
        }
    });

    add_action( 'plugins_loaded', 'init_kopokopo_class' );
    add_action('woocommerce_checkout_init','disable_billing');

    function disable_billing( $checkout ) {
        $checkout->checkout_fields['billing'] = array();
        return $checkout;
    }

    function init_kopokopo_class() {
        class WC_Paytalk_KopoKopo_Gateway extends WC_Payment_Gateway {
            function __construct() {
                $this->id                 = 'kopokopo';
                $this->method_title       = __('PayTalk.co.ke(KopoKopo)', 'woocommerce');
                $this->method_description = __('Paytalk KopoKopo gateway works by adding form fields on the checkout page and then sending the details to Paytalk.co.ke for verification and processing. Get API keys from <a href="https://app.kopokopo.com" target="_blank">https://app.kopokopo.com</a>', 'woocommerce');
                $this->icon               = plugins_url( '/images/paytalk_160x68.png', __FILE__ );
                $this->has_fields         = true;
                $this->supports           = array( 'products' );
                
                $this->liveurl            = 'https://developer.paytalk.co.ke/api/';

                $this->init_form_fields();
                $this->init_settings();

                $this->title       = "Lipa Na Mpesa";
                $this->description = $this->settings['description'];
                $this->enabled     = $this->settings['enabled'];
                $this->stk_push    = $this->settings['stk_push'];
                $this->sms         = $this->settings['sms'];
                $this->paytill     = $this->settings['paytill'];
                $this->client_id   = $this->settings['client_id'];
                $this->client_secret = $this->settings['client_secret'];
                $this->opa         = $this->settings['opa'];

                add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( &$this, 'process_admin_options' ) );
            }

            function init_form_fields() {

				$this->form_fields = array(
					/*'title' => array(
			            'title'       => __( 'Title', 'woocommerce' ),
			            'type'        => 'text',
			            'description' => __( 'This controls the title for the payment method the customer sees during checkout.', 'woocommerce' ),
			            'default'     => __( 'Paytalk.co.ke', 'wc-gateway-offline' ),
			            'desc_tip'    => true,
			        	),*/

					'enabled' => array(
						'title'       => __( 'Enable/Disable', 'woocommerce' ),
						'label'       => __( 'Enable KopoKopo', 'woocommerce' ),
						'type'        => 'checkbox',
						'description' => '',
						'default'     => 'no'
						),

					'stk_push' => array(
						'title'       => __( 'Enable/Disable', 'woocommerce' ),
						'label'       => __( 'Enable KopoKopo K2 STK Push', 'woocommerce' ),
						'type'        => 'checkbox',
						'description' => __( 'Enable KopoKopo K2 STK Push', 'woocommerce' ),
						'default'     => 'no',
						'desc_tip'    => true
						),

					'sms' => array(
						'title'       => __( 'Enable/Disable SMS', 'woocommerce' ),
						'label'       => __( 'Enable SMS notification', 'woocommerce' ),
						'type'        => 'checkbox',
						'description' => __( 'Receive/Send SMS with your own sender ID. Contact info@paytalk.co.ke to setup your SMS account.', 'woocommerce' ),
						'default'     => 'no',
						//'desc_tip'    => true
						),

					'paytill' => array(
						'title'       => __( 'KopoKopo Paybill/Till Number', 'woocommerce' ),
						'type'        => 'text',
						'description' => '',
						'default'     => ''
						),

					'description' => array(
						'title'       => __( 'Description', 'woocommerce' ),
						'type'        => 'textarea',
						'description' => __( 'This controls the description which the user sees during checkout.', 'woocommerce' ),
						'default'     => 'Pay with Till Number/Paybill via PayTalk.',
						'desc_tip'    => true
						),
					
					'client_id' => array(
						'title'       => __( 'KopoKopo Client ID', 'woocommerce' ),
						'type'        => 'password',
						'description' => sprintf( __( 'Get your KopoKopo Client ID from KopoKopo <a href="%s" target="_blank">KopoKopo</a>', 'woocommerce' ), 'https://app.kopokopo.com/' ),
						'default'     => ''
						),

					'client_secret' => array(
						'title'       => __( 'KopoKopo Client Secret', 'woocommerce' ),
						'type'        => 'password',
						'description' => sprintf( __( 'Get your client secret from KopoKopo <a href="%s" target="_blank">KopoKopo</a>', 'woocommerce' ), 'https://app.kopokopo.com/' ),
						'default'     => '',
						'placeholder' => ''
						),

					'opa' => array(
						'title'       => __( 'Online Payment Account', 'woocommerce' ),
						'type'        => 'text',
						'description' => sprintf( __( 'Get your Online Payment Account from KopoKopo <a href="%s" target="_blank">KopoKopo</a>', 'woocommerce' ), 'https://app.kopokopo.com/' ),
						'default'     => '',
						'placeholder' => 'Online Payment Account starts with a K eg. K123456'
						),


				);

			}

            //form fields & payment_fields
            public function payment_fields() {
				if ($description = $this->get_description()) {
					echo wpautop(wptexturize($description));
				}
				$stk_push = ( $this->stk_push == "yes" ) ? 'TRUE' : 'FALSE';
				$sms = ( $this->sms == "yes" ) ? 'TRUE' : 'FALSE';
				if($stk_push == "FALSE"){
				?> 
							<div style="max-width:300px"> 
							<p class="form-row form-row form-row-wide woocommerce-validated" id="mpesa_phone_field" data-o_class="form-row form-row form-row-wide">
								<label for="mpesa_phone" class="">Phone Number <abbr class="required" title="required">*</abbr></label>
								<input type="text" class="input-text" name="mpesa_phone" id="mpesa_phone" placeholder="Phone Number" required />
							</p>
							<p class="form-row form-row form-row-wide woocommerce-validated" id="mpesa_code_field" data-o_class="form-row form-row form-row-wide">
								<label for="mpesa_code" class="">Transaction ID <abbr class="required" title="required">*</abbr></label>
								<input type="text" class="input-text" name="mpesa_code" id="mpesa_code" placeholder="Transaction ID" />
							</p>
							</div>
				<?php
					}else{
				?>
							<div style="max-width:300px"> 
							<p class="form-row form-row form-row-wide woocommerce-validated" id="mpesa_phone_field" data-o_class="form-row form-row form-row-wide">
								<label for="mpesa_phone" class="">Phone Number <abbr class="required" title="required">*</abbr></label>
								<input type="text" class="input-text" name="mpesa_phone" id="mpesa_phone" placeholder="Phone Number" required />
							</p>
							</div>
				<?php
					}
				}

			public function validate_fields() { 
				if($stk_push == "FALSE"){
					if ($_POST['mpesa_phone']) {
						$success = true;
					} else {					
						$error_message = __("The ", 'woothemes') . $this->field_title . __(" Phone Number is required", 'woothemes');
						wc_add_notice(__('Field error: ', 'woothemes') . $error_message, 'error');
						$success = False;
					}

					if ($_POST['mpesa_code']) {
						$success = true;
					} else {					
						$error_message = __("The ", 'woothemes') . $this->phone_title . __(" Transaction ID is required", 'woothemes');
						wc_add_notice(__('Field error: ', 'woothemes') . $error_message, 'error');
						$success = False;
					}
					return $success;
		
				}else{
					if ($_POST['mpesa_phone']) {
						$success = true;
					} else {					
						$error_message = __("The ", 'woothemes') . $this->field_title . __(" Phone Number is required", 'woothemes');
						wc_add_notice(__('Field error: ', 'woothemes') . $error_message, 'error');
						$success = False;
					}
				}
			}


            public function process_payment( $order_id ) {
                global $woocommerce;

                $customer_order = wc_get_order( $order_id ); //HPOS safe
                
                $environment_url = 'https://developer.paytalk.co.ke/api/kopokopo/';

                // use wc_clean instead of woocommerce_clean
                $mpesa_phone = isset($_POST['mpesa_phone']) ? wc_clean($_POST['mpesa_phone']) : '';
                $mpesa_code  = isset($_POST['mpesa_code']) ? wc_clean($_POST['mpesa_code']) : '';

                $items = $woocommerce->cart->get_cart();
                $order_items = array();
                foreach($items as $item => $values) { 
                    $_product = wc_get_product( $values['data']->get_id()); 
                    $order_items[] = "<b> ".$_product->get_name().'</b>  <br> Quantity: '.$values['quantity'].'<br> Price: '.get_post_meta($values['product_id'], '_price', true);
                } 

                $payload = [
                    "paytill"      => $this->paytill,
                    "x_version"    => "3.1.0",
                    "x_amount"     => $customer_order->get_total(), // getter
                    "mpesa_code"   => $mpesa_code,
                    "mpesa_phone"  => $mpesa_phone,
                    "stk_push"     => $this->stk_push,
                    "opa"          => $this->opa,
                    "klient_id"    => $this->client_id,
                    "klient_sec"   => $this->client_secret,
                    "x_type"       => 'AUTH_CAPTURE',
                    "x_invoice_num"=> str_replace( "#", "", $customer_order->get_order_number() ),
                    "order_id"     => $customer_order->get_id(),
                    "items"        => $order_items,
                    "x_first_name" => $customer_order->get_billing_first_name(),
                    "x_last_name"  => $customer_order->get_billing_last_name(),
                    "x_address"    => $customer_order->get_billing_address_1(),
                    "x_city"       => $customer_order->get_billing_city(),
                    "x_state"      => $customer_order->get_billing_state(),
                    "x_zip"        => $customer_order->get_billing_postcode(),
                    "x_country"    => $customer_order->get_billing_country(),
                    "x_phone"      => $customer_order->get_billing_phone(),
                    "x_email"      => $customer_order->get_billing_email(),
                    "x_cust_id"    => $customer_order->get_user_id(),
                    "x_customer_ip"=> $_SERVER['REMOTE_ADDR'],
                    "x_url"        => home_url(),
                    "x_sms"        => $this->sms
                ];

                $response = wp_remote_post( $environment_url, array(
                    'method'    => 'POST',
                    'body'      => http_build_query( $payload ),
                    'timeout'   => 90,
                    'sslverify' => true,
                ));

                // your same error checks remain here...
                if ( is_wp_error( $response ) ) 
                    throw new Exception( __( 'We are currently experiencing problems trying to connect to PayTalk.co.ke. Sorry for the inconvenience.', 'woocommerce' ) );

                if ( empty( $response['body'] ) )
                    throw new Exception( __( 'PayTalk\'s Response was empty.', 'woocommerce' ) );

                if ( $response['body'] == "no_account" )
                    throw new Exception( __( 'Make sure you are using valid PayTalk credentials and Paybill/Till Number.', 'woocommerce' ) );

                if ( $response['body'] == "used_trans" )
                    throw new Exception( __( 'The Transaction ID has already been used. Please try again.', 'woocommerce' ) );

                if ( is_numeric($response['body']) )
                    throw new Exception( __( 'Payment was short by Ksh'.$response['body'].'. Kindly pay the balance to complete your order.', 'woocommerce' ) );

                if ( $response['body'] == "no_trans" )
                    throw new Exception( __( 'We could not verify your payment. Please check your phone.', 'woocommerce' ) );

                if ( $response['body'] == "stk_fail" )
                    throw new Exception( __( 'Unable to initiate payment, please try again later.', 'woocommerce' ) );

                if ( $response['body'] == "phone_err" )
                    throw new Exception( __( 'Invalid phone number.', 'woocommerce' ) );

                if ( $response['body'] == "Success" ) {
                    $customer_order->add_order_note( __( 'Paytalk.co.ke payment completed.', 'woocommerce' ) );
                    $customer_order->payment_complete();
                    wc_reduce_stock_levels( $customer_order->get_id() ); // HPOS safe
                    $woocommerce->cart->empty_cart();
                    return array(
                        'result'   => 'success',
                        'redirect' => $this->get_return_url( $customer_order ),
                    );
                }

                if ( $response['body'] == "offline" ) {
                    $customer_order->add_order_note( __( 'Paytalk offline Payment - Awaiting confirmation.', 'woocommerce' ) );
                    $customer_order->update_status('on-hold', __( 'Paytalk offline Payment - Awaiting confirmation.', 'woocommerce' ));
                    wc_reduce_stock_levels( $customer_order->get_id() ); //HPOS safe
                    $woocommerce->cart->empty_cart();
                    return array(
                        'result'   => 'success',
                        'redirect' => $this->get_return_url( $customer_order ),
                    );
                }

                // If we reached here → error
                wc_add_notice( __( 'Payment failed. Please try again.', 'woocommerce' ), 'error' );
                $customer_order->add_order_note( 'Error: Unexpected response from Paytalk - ' . $response['body'] );
            }
        }

        function add_init_kopokopo_paytalk_class($methods) {
            $methods[] = 'WC_Paytalk_KopoKopo_Gateway'; 
            return $methods;
        }
        add_filter('woocommerce_payment_gateways', 'add_init_kopokopo_paytalk_class');
    }
} else {
    function my_error_notice() {
        ?>
        <div class="error notice">
            <p><?php _e( '<b>KopoKopo Lipa Na Mpesa Paytalk gateway requires WooCommerce to be activated</b>', 'woocommerce' ); ?></p>
        </div>
        <?php
    }
    add_action( 'admin_notices', 'my_error_notice' );
}
